#' @title Calculates the pathway-based mutation accumulate perturbation score
#' @description The function `get_pfs_score` uses to calculate the pathway-based mutation accumulate perturbation score using the matrix of gene mutation state and pathway information.
#' @param mut_status Mutation status of a particular gene in a particular sample. The file can be generated by the function `get_mut_status`.
#' @param percent This parameter is used to control the mutation rate of gene. Genes less than this value will be deleted
#' @param gene_Ucox_res Results of gene univariate Cox regression.
#' @param data.dir Location of the "organism"SPIA.RData file containing the pathways data.If set to NULL will look for this file in the extdata folder of the PFS library.
#' @param organism A three letter character designating the organism. See a full list at ftp://ftp.genome.jp/pub/kegg/xml/organisms.
#' @param verbose If set to TRUE, displays the number of pathways already analyzed.
#' @param gene_symbol_Entrez A data table containing gene symbol and gene Entrez ID.
#' @param Entrez_ID Logical,tell whether there are Entrez IDs corresponding to gene symbol in the gene_symbol_Entrez.
#' @param gene_set A group of cancer specific gene symbols obtained from the training set
#' @return A binary mutations matrix, which column names is sample and the row name is the pathway.
#' @importFrom utils read.delim
#' @export
#' @examples
#' #get the path of the mutation annotation file.
#' data(mut_status,gene_Ucox_res,gene_symbol_Entrez)
#' #perform the function `get_pfs_score`.
#' \donttest{pfs_score<-get_pfs_score(mut_status[,1:2],percent=0.03,gene_Ucox_res,gene_symbol_Entrez)}


get_pfs_score<-function(mut_status,percent,gene_Ucox_res,gene_symbol_Entrez,data.dir=NULL,organism="hsa",
                        verbose=TRUE,Entrez_ID=TRUE,gene_set=NULL){
  rownames(mut_status)<-gsub(pattern = "-",replacement = ".",rownames(mut_status))
  if(is.null(gene_set)){
    a<-apply(mut_status,1,function(x){(length(which(x!=0))/length(x))})
    mut_status<-cbind(mut_status,a)
    mut_status<-mut_status[-which(mut_status[,dim(mut_status)[2]]<percent),]
    mut_status<-mut_status[,-dim(mut_status)[2]]
    mut_status<-mut_status[match(rownames(gene_Ucox_res)[which(gene_Ucox_res$HR<1)],rownames(mut_status)),]
  }else{
    mut_status<-mut_status[match(gene_set,rownames(mut_status)),]
  }
  mut_status<-get_Entrez_ID(mut_status,gene_symbol_Entrez,Entrez_ID=Entrez_ID)
  .myDataEnv <- new.env(parent = emptyenv())
  datload <- paste(organism, "SPIA", sep = "")
  if (is.null(data.dir)) {
    if (!paste(datload, ".RData", sep = "") %in%
        dir(system.file("extdata", package = "PMAPscore"))) {
      cat("The KEGG pathway data for your organism is not present in the extdata folder of the SPIA package!!!")
      cat("\n")
      cat("Please generate one first using makeSPIAdata and specify its location using data.dir argument or copy it in the extdata folder of the SPIA package!")
    }
    else {
      load(file = paste(system.file("extdata", package = "PMAPscore"),
                        paste("/", organism, "SPIA", sep = ""),
                        ".RData", sep = ""), envir = .myDataEnv)
    }
  }
  if (!is.null(data.dir)) {
    if (!paste(datload, ".RData", sep = "") %in%
        dir(data.dir)) {
      cat(paste(data.dir, " does not contin a file called ",
                paste(datload, ".RData", sep = "")))
    }
    else {
      load(file = paste(data.dir, paste(datload, ".RData",
                                        sep = ""), sep = ""), envir = .myDataEnv)
    }
  }
  path.info = .myDataEnv[["path.info"]]
  newlist<-list()
  for(i in 1:dim(mut_status)[2])
  {
    path_matrix<-as.numeric(mut_status[,i])
    names(path_matrix)<-rownames(mut_status)
    path_matrix<-path_matrix[which(path_matrix!=0)]
    if(length(path_matrix[which(path_matrix!=0)])>0){
      res<-newspia(de=path_matrix[which(path_matrix!=0)],all=rownames(mut_status),organism="hsa",
                   beta=NULL,verbose=verbose,data.dir=data.dir,pathids = NULL)
    }
    newlist[[i]]<-res
    names(newlist)[[i]]<-colnames(mut_status)[i]
  }
  pathname<-c()
  for (j in 1:length(path.info)) {
    pathname<-c(pathname,path.info[[j]]$title)
  }
  pfs_score<-matrix(data=0,nrow = length(pathname),ncol = dim(mut_status)[2])
  rownames(pfs_score)<-pathname
  colnames(pfs_score)<-colnames(mut_status)
  for(a in 1:length(newlist))
  {
    loc<-match(names(newlist)[[a]],colnames(pfs_score))
    loc2<-match(newlist[[a]][,1],rownames(pfs_score))
    pfs_score[loc2,loc]<-newlist[[a]][,3]
  }
  return(pfs_score)
}
