% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SCB_functional_outcome.R
\name{SCB_functional_outcome}
\alias{SCB_functional_outcome}
\title{Construct Simultaneous Confidence Bands (SCB) For One Dimensional Functional Data}
\usage{
SCB_functional_outcome(
  data_df,
  object = NULL,
  method,
  fitted = TRUE,
  alpha = 0.05,
  outcome,
  domain,
  subset = NULL,
  id,
  nboot = NULL,
  method_SD = "t",
  weights = "rademacher"
)
}
\arguments{
\item{data_df}{A functional data frame that contains both name and values for
variables including functional outcome, domain (e.g. time) and ID (e.g. subject names)
used to fit the model \code{object}.}

\item{object}{A fitted Function-on-Scalar Regression (FoSR) object
(e.g., from mgcv::gam()/mgcv::bam()). Default is \code{NULL}}

\item{method}{A character string specifying the approach:
\itemize{
\item \code{"cma"} - Correlation and Multiplicity Adjusted (CMA) confidence bands
via parametric approach (requires a fitted functional regression model)
\item \code{"multiplier"} - Dense confidence bands via Multiplier-t Bootstrap method
For \code{method = "multiplier"}, the outcome variable in \code{data_df}
should not have all-zero entries within any specified domain
(except for domain index zero, where this is allowed).
Otherwise, the function will return an error.
If missing values (NA) exist in the outcome variable in \code{data_df},
the function will impute them using \code{fpca.face} before
performing the Multiplier Bootstrap.
}}

\item{fitted}{Logical. Whether to estimate the simultaneous confidence bands
for the fitted mean function or the fitted parameter function
\itemize{
\item \code{TRUE} - Estimate the simultaneous confidence bands
for the fitted mean outcome function.
\item \code{FALSE} - estimate the simultaneous confidence bands
for the fitted parameter function.
}
Default is \code{TRUE}.}

\item{alpha}{Significance level for SCB. Default is 0.05.}

\item{outcome}{A character string specifying the name of the outcome variable
used in the model.}

\item{domain}{A character string specifying the name of the domain variable
(e.g. time) used in the model.}

\item{subset}{An atomic character vector (e.g., c("user = 1", "age = 30"))
specifying the target function for constructing the SCB.
Each element must be of the form \verb{<name> = <value>}, where \verb{<name>} is
the name of a scalar grouping variable and \verb{<value>} is the desired value.
Whitespace is ignored. Binary or categorical character variables should be
transformed into numeric. Factors are not allowed here because if the input
data contains factor variables, they will be automatically expanded into
dummy (indicator) variables when constructing the design matrix, and
the resulting variable names may differ from the original factor names.
Default is \code{NULL}, representing the reference group.}

\item{id}{A character string specifying the name of the ID variable.}

\item{nboot}{An integer specifying the number of bootstrap samples used to
construct the confidence bands. Default is 10,000 for cma, 5000 for Multiplier Bootstrap.}

\item{method_SD}{Method for SD estimation: "t" or "regular". Default is "t".}

\item{weights}{Multiplier type: "rademacher", "gaussian", or "mammen".
Default is "rademacher".}
}
\value{
A list containing:
\describe{
\item{mu_hat}{Estimated mean function for the group of interest.}
\item{domain}{The domain used.}
\item{se_hat}{Standard errors of the estimated means.}
\item{scb_low}{Lower bound of the simultaneous confidence band.}
\item{scb_up}{Upper bound of the simultaneous confidence band.}
\item{type}{A character description of the output type.}
}
}
\description{
This function builds simultaneous confidence bands through parametric and
bootstrap approaches.
}
\examples{
# example using pupil data
if (requireNamespace("mgcv", quietly = TRUE)) {
data(pupil)
\donttest{
pupil_fpca <- prepare_pupil_fpca(pupil)

fosr_mod <- mgcv::bam(percent_change ~ s(seconds, k=30, bs="cr") +
  s(seconds, by = use, k=30, bs = "cr") +
  s(id, by = Phi1, bs="re") +
  s(id, by = Phi2, bs="re"),
  method = "fREML", data = pupil_fpca, discrete = TRUE)

# CMA approach
results <- SCB_functional_outcome(data_df = pupil, object = fosr_mod,
                                  method = "cma", fitted = TRUE,
                                  outcome = "percent_change", domain = "seconds",
                                  subset = c("use = 1"), id = "id")


# multiplier bootstrap
results <- SCB_functional_outcome(data_df = pupil, object = fosr_mod,
                                  method = "multiplier", fitted = TRUE,
                                  outcome = "percent_change", domain = "seconds",
                                  subset = c("use = 1"), id = "id")
}

mean_mod <- mgcv::gam(percent_change ~ s(seconds, k = 5, bs = "cr") +
s(seconds, by = use, k = 5, bs = "cr"),
data = pupil, method = "REML")

# multiplier bootstrap
pupil_multiplier <- SCB_functional_outcome(data = pupil, object = mean_mod, method = "multiplier",
                                   outcome = "percent_change",
                                   domain = "seconds", subset= c("use = 1"),
                                   id = "id")
}

}
