#' Creates Uda-plot for the provided data with and without the estimated normal density curve, excluding extreme values.
#'
#' @description Uda-plot can be utilized to assess normality.
#' @usage udaplot(X, excld = FALSE, npdf = FALSE, title = ifelse(npdf == FALSE, "Uda-plot",
#'        "Uda-plot & Normal Density Curve"), xlab = "x", lcol = "black", rcol = "grey60",
#'        pdfcol = "red",...)
#' @param X an \eqn{n} by \eqn{1} matrix, equivalently, a column vector of length \eqn{n}, where \eqn{n} is the number of observations.
#' @param excld exclude extreme values in the data applying 1.5IQR rule, \emph{FALSE} by default.
#' @param npdf display of the estimated normal density curve in the Uda-plot, \emph{FALSE} by default.
#' @param title title of the plot, \emph{Uda-plot} by default and \emph{Uda-plot & Normal Density Curve} if npdf is \emph{TRUE}.
#' @param xlab \eqn{x}-axis label, \eqn{x} by default.
#' @param lcol color of the points which correspond to data on or below the sample average, \emph{black} by default.
#' @param rcol color of the points which correspond to data above the sample average, \emph{grey60} by default.
#' @param pdfcol color of the estimated normal density curve, \emph{red} by default.
#' @param ... other graphical parameters.
#'
#' @import ggplot2
#' @return Uda-plot
#' @references Wijesuriya, U. A. (2025a). Ada-plot and Uda-plot as Alternatives for
#'             Ad-plot and Ud-plot.
#'             \emph{Communications in Statistics-Theory and Methods}, \doi{10.1080/03610926.2025.2558108}.
#' @references Wijesuriya, U. A. (2025b). Ad-plot and Ud-plot for Determining Distributional Characteristics
#'             and Normality.
#'             \emph{Communications in Statistics-Theory and Methods}, \doi{10.1080/03610926.2024.2440583}.
#' @references Wijesuriya, U. A. (2025c). adplots: Ad-plot and Ud-plot for Visualizing Distributional
#'             Properties and Normality.
#'             \emph{R Package Version 0.1.0}, \doi{10.32614/CRAN.package.adplots}.
#' @examples
#'    set.seed(0)
#'    X1 <- matrix(c(rnorm(50, mean = 2, sd = 5), runif(4, 17, 30)))
#'    udaplot(X1)
#'
#'    X2 <- matrix(c(rnorm(50, mean = 2, sd = 5), runif(4, 17, 30)))
#'    udaplot(X2, npdf = TRUE)
#'
#'    X3 <- matrix(c(rnorm(50, mean = 2, sd = 5), runif(4, 17, 30)))
#'    udaplot(X3, excld = TRUE, npdf = TRUE)
#'
#'    X4 <- matrix(rnorm(500, mean = 2, sd = 5))
#'    udaplot(X4, title = "", lcol = "blue", rcol = "red", pdfcol = "balck")
#' @export
udaplot <- function(X, excld = FALSE, npdf = FALSE, title = ifelse(npdf == FALSE, "Uda-plot", "Uda-plot & Normal Density Curve"), xlab = "x", lcol = "black", rcol = "grey60", pdfcol = "red",...){

  if (excld == FALSE)
  {
    D <- sort(X)                                                       # Sorts data in ascending order
    n <- length(D)                                                     # Computes sample size
    xbar <- mean(D)                                                    # Computes sample average
    s <- sd(D)                                                         # Computes sample standard deviation
    abdev <- abs(D-xbar)                                               # Computes absolute deviation
    ylab = bquote('W'[.(n)](x))                                        # Provides y-axis label

    if (npdf == FALSE)
    {
      U <- matrix(NA, nrow = n, ncol = 1)                              # Defines an NA column vector size n
      for (j in 1:n){
        U[j] <- round((1/sqrt(2*pi*s^2))-(1/(2*s^2*(1-n^(-1))))*(1/n)*sum(abdev[abdev<=abs(xbar-D[j])]),10) # Computes values of the Centralized Empirical Cumulative Average Deviation Function, following Theorem 3
      }
      df <- data.frame(D,U)                                                # Creates a dataframe
      g <- ggplot(df, aes(x = D))+
        geom_point(aes(x = D, y = U), col = ifelse(D <= xbar, lcol, rcol))+
        labs(title = title)+
        xlab(xlab)+
        ylab(ylab)+
        theme(axis.text = element_text(size = 12), axis.title = element_text(size = 12))  # Creates the ggplot
    }

    if (npdf==TRUE)
    {
      U <- matrix(NA, nrow = n, ncol = 1)                               # Defines an NA column vector size n
      f <- matrix(NA, nrow = n, ncol = 1)                               # Defines an NA column vector size n
      q <- matrix(NA, nrow = n, ncol = 1)                               # Defines an NA column vector size n
      for (j in 1:n){
        U[j] <- round((1/sqrt(2*pi*s^2))-(1/(2*s^2*(1-n^(-1))))*(1/n)*sum(abdev[abdev<=abs(xbar-D[j])]),10) # Computes values of the Centralized Empirical Cumulative Average Deviation Function, following Theorem 3
        f[j] <- round((1/sqrt(2*pi*s^2))*exp(-(1/(2*s^2))*(D[j] - xbar)^2), 10)           # Computes the normal density estimating the population mean and standard deviation by sample average and sample standard deviation, respectively
        q[j] <- round(min(U[j], f[j])/max(U[j], f[j]), 4)
      }
      d <- round(sum(q)/n, 2)                                           # Computes the d-value
      df <- data.frame(D, U, f)                                         # Creates a dataframe
      g <- ggplot(df, aes(x = D))+
        geom_point(aes(x = D, y = U), col = ifelse(D <= xbar, lcol, rcol))+
        stat_function(fun = dnorm, args=list(mean = xbar,sd = s), colour = pdfcol)+
        labs(title = title)+
        xlab(xlab)+
        ylab(ylab)+
        annotate(geom = "label",x = Inf, y = Inf, parse = TRUE, col = "black", label = paste0('italic(d)-value ==', d), vjust=1.5, hjust = 1.1, size=4.5, fill = "grey92", label.size = NA)+
        theme(axis.text = element_text(size = 12),axis.title = element_text(size = 12))    # Creates the ggplot
    }
  }

  if (excld == TRUE)                                                 # Sorts data in ascending order
  {
    D <- sort(X)                                                     # Sorts data in ascending order
    Q1 <- quantile(D)[["25%"]]                                       # Finds the first quartile
    Q3 <- quantile(D)[["75%"]]                                       # Finds the third quartile
    IQR <- Q3-Q1                                                     # Finds the interquartile range
    L <- Q1-1.5*IQR                                                  # Defines the lower bound
    U <- Q3+1.5*IQR                                                  # Defines the upper bound
    Dr <- {}
    Dr <- D[D>=L & D<=U]                                             # Excludes extreme values followin thhe 1.5IQR rule
    n <- length(Dr)
    D <- matrix(NA, nrow = n, ncol = 1)                              # Defines an NA column vector size n
    D <- Dr
    xbar <- mean(D)                                                    # Computes sample average
    s <- sd(D)                                                         # Computes sample standard deviation
    abdev <- abs(D-xbar)                                               # Computes absolute deviation
    ylab = bquote('W'[.(n)](x))                                        # Provides y-axis label

    if (npdf == FALSE)
    {
      U <- matrix(NA, nrow = n, ncol = 1)                              # Defines an NA column vector size n
      for (j in 1:n){
        U[j] <- round((1/sqrt(2*pi*s^2))-(1/(2*s^2*(1-n^(-1))))*(1/n)*sum(abdev[abdev<=abs(xbar-D[j])]),10) # Computes values of the Centralized Empirical Cumulative Average Deviation Function, following Theorem 3
      }
      df <- data.frame(D,U)                                                # Creates a dataframe
      g <- ggplot(df, aes(x = D))+
        geom_point(aes(x = D, y = U), col = ifelse(D <= xbar, lcol, rcol))+
        labs(title = title)+
        xlab(xlab)+
        ylab(ylab)+
        theme(axis.text = element_text(size = 12), axis.title = element_text(size = 12))  # Creates the ggplot
    }

    if (npdf==TRUE)
    {
      U <- matrix(NA, nrow = n, ncol = 1)                               # Defines an NA column vector size n
      f <- matrix(NA, nrow = n, ncol = 1)                               # Defines an NA column vector size n
      q <- matrix(NA, nrow = n, ncol = 1)                               # Defines an NA column vector size n
      for (j in 1:n){
        U[j] <- round((1/sqrt(2*pi*s^2))-(1/(2*s^2*(1-n^(-1))))*(1/n)*sum(abdev[abdev<=abs(xbar-D[j])]),10) # Computes values of the Centralized Empirical Cumulative Average Deviation Function, following Theorem 3
        f[j] <- round((1/sqrt(2*pi*s^2))*exp(-(1/(2*s^2))*(D[j] - xbar)^2), 10)           # Computes the normal density estimating the population mean and standard deviation by sample average and sample standard deviation, respectively
        q[j] <- round(min(U[j], f[j])/max(U[j], f[j]), 4)
      }
      d <- round(sum(q)/n, 2)                                           # Computes the d-value
      df <- data.frame(D, U, f)                                         # Creates a dataframe
      g <- ggplot(df, aes(x = D))+
        geom_point(aes(x = D, y = U), col = ifelse(D <= xbar, lcol, rcol))+
        stat_function(fun = dnorm, args=list(mean = xbar,sd = s), colour = pdfcol)+
        labs(title = title)+
        xlab(xlab)+
        ylab(ylab)+
        annotate(geom = "label",x = Inf, y = Inf, parse = TRUE, col = "black", label = paste0('italic(d)-value ==', d), vjust=1.5, hjust = 1.1, size=4.5, fill = "grey92", label.size = NA)+
        theme(axis.text = element_text(size = 12),axis.title = element_text(size = 12))    # Creates the ggplot
    }
  }
  return(g)
}
