#' Plot function.
#'
#' @description Plots objects of class `testmc`.
#' Supports global and local types, multiple marks, and fully customizable appearance via `layers` or `...`.
#'
#' @param x An object of class `testmc` or `global_envelope`.
#' @param ... Ignored.
#' @section Additional graphical customization:
#' The object returned by \code{plot.testmc()} is a \code{ggplot} object. This means
#' that all standard graphical components from the \pkg{ggplot2} package (such as
#' layers, scales, and themes) can be added to the plot using the \code{+} operator.
#' This allows users to further customize labels, colors, axes, and the overall
#' appearance of the plot.
#' @return Invisibly returns the ggplot object.
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com}
#' 
#' 
#' @examples
#' \donttest{
#'  library(spatstat.geom)
#'  library(spatstat.random)
#'  library(spatstat.explore)
#'  library(spatstat.linnet)
#'  library(ggplot2)
#'
#' #######################################
#' ## 1. Planar point patterns (ppp)
#' #######################################
#'
#' # Global mark correlation (real-valued marks)
#' X_ppp <- rpoispp(200)
#' marks(X_ppp) <- data.frame(m1 = runif(npoints(X_ppp), 1, 10))
#'
#' tc_ppp1 <- testmc(X_ppp,
#'                    fun = mcorr.ppp,
#'                    fun_args = list(ftype = "stoyan", method = "density"))
#'
#' plot(tc_ppp1) +
#'   labs(
#'     title = "Global mark correlation (ppp)",
#'     x = expression(r),
#'     y = "Observed / Envelope"
#'   ) +
#'   theme_minimal() +
#'   theme(
#'     plot.title = element_text(size = 14, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text = element_text(size = 10),
#'     legend.title = element_text(size = 11),
#'     legend.text = element_text(size = 10)
#'   )
#'
#' # Local mark correlation (real-valued marks)
#' X_ppp2 <- rpoispp(200)
#' marks(X_ppp2) <- data.frame(m1 = runif(npoints(X_ppp2), 1, 10))
#'
#' tc_ppp2 <- testmc(X_ppp2,
#'                    fun = lmcorr.ppp,
#'                    fun_args = list(ftype = "stoyan", method = "density"))
#'
#' plot(tc_ppp2[[1]]) +
#'   labs(
#'     title = "Local mark correlation (ppp)",
#'     x = expression(r),
#'     y = "Local correlation"
#'   ) +
#'   theme_classic() +
#'   theme(
#'     plot.title = element_text(size = 13, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text = element_text(size = 10),
#'     legend.title = element_text(size = 11),
#'     legend.text = element_text(size = 10),
#'     legend.position = "right"
#'   )
#'
#' plot(tc_ppp2) +
#'   labs(
#'     title = "Effective range",
#'     x = expression(r),
#'     y = "Id of significant points"
#'   ) +
#'   theme_classic() +
#'   theme(
#'     plot.title = element_text(size = 23, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text = element_text(size = 20),
#'     legend.title = element_text(size = 21),
#'     legend.text = element_text(size = 30),
#'     legend.position = "right"
#'   )
#'
#' #######################################
#' ## 2. Linear network point patterns (lpp)
#' #######################################
#'
#' X_lpp <- rpoislpp(40, simplenet)
#' marks(X_lpp) <- data.frame(m1 = runif(npoints(X_lpp), 1, 10))
#'
#' tc_lpp1 <- testmc(X_lpp,
#'                   fun = mcorr.lpp,
#'                   fun_args = list(ftype = "stoyan", method = "density"))
#'
#' plot(tc_lpp1) +
#'   labs(
#'     title = "Global mark correlation (lpp)",
#'     x = expression(r),
#'     y = "Observed / Envelope"
#'   ) +
#'   theme_bw() +
#'   theme(
#'     plot.title = element_text(size = 14, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text = element_text(size = 10),
#'     legend.title = element_text(size = 11),
#'     legend.text = element_text(size = 10),
#'     legend.position = "top"
#'   )
#'
#' # Local functional mark correlation (function-valued marks)
#' marks(X_lpp) <- data.frame(
#'   t1 = runif(npoints(X_lpp), 1, 10),
#'   t2 = runif(npoints(X_lpp), 1, 10)
#' )
#'
#' tc_lpp2 <- testmc(X_lpp,
#'                   fun = lfmcorr,
#'                   fun_args = list(ftype = "stoyan", method = "density"))
#'
#' plot(tc_lpp2[[1]]) +
#'   labs(
#'     title = "Local functional mark correlation (lpp)",
#'     x = expression(r),
#'     y = "Local correlation"
#'   ) +
#'   theme_light() +
#'   theme(
#'     plot.title = element_text(size = 13, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text = element_text(size = 10),
#'     legend.title = element_text(size = 11),
#'     legend.text = element_text(size = 10),
#'     legend.position = "bottom"
#'   )
#'
#' }
#' @export

plot.testmc <- function(x, ...) {
  
  scenario3_funs <- c("lmcorr.ppp", "lmcorr.lpp", "lmcorrinhom.ppp",
                      "lmcorrinhom.lpp", "lfmcorr", "lfmcorrinhom")
  
  fun_attr <- attr(x, "fun")
  
  # ---- Helper: get contiguous intervals ----
  get_timestamp_ranges <- function(timestamps, threshold) {
    if(length(timestamps) == 0) return(matrix(numeric(0), ncol=2))
    segments <- split(timestamps, cumsum(c(1, diff(timestamps) > threshold)))
    do.call(rbind, lapply(segments, range))
  }
  
  # ---- Scenario 1: single global_envelope ----
  if(inherits(x, "global_envelope")) {
    df <- as.data.frame(x)
    p <- ggplot2::ggplot(df) +
      ggplot2::geom_ribbon(ggplot2::aes(x = r, ymin = lo, ymax = hi), fill = "grey80") +
      ggplot2::geom_line(ggplot2::aes(x = r, y = obs), color = "black") +
      ggplot2::geom_line(ggplot2::aes(x = r, y = central), linetype = "dashed", color = "black") +
      ggplot2::theme_minimal()  # <-- same background as other scenarios
    return(p)
  }
  
  # ---- Scenario 2: list of global_envelope objects ----
  if((is.list(x) && all(sapply(x, inherits, "global_envelope"))) && !any(fun_attr == scenario3_funs) && fun_attr !="fun") {
    plots <- lapply(x, function(ge) {
      df <- as.data.frame(ge)
      ggplot2::ggplot(df) +
        ggplot2::geom_ribbon(ggplot2::aes(x = r, ymin = lo, ymax = hi), fill = "grey80") +
        ggplot2:: geom_line(ggplot2::aes(x = r, y = obs), color = "black") +
        ggplot2:: geom_line(ggplot2::aes(x = r, y = central), linetype = "dashed", color = "black") +
        ggplot2::theme_minimal()  # <-- unified background
    })
    
    combined <- Reduce(`+`, plots) + patchwork::plot_layout(ncol = length(plots))
    return(combined)
  }
  
  # ---- Scenario 3: list of lists of global_envelope objects ----
  if(is.list(x) && all(sapply(x, is.list))) {
    
    if(is.null(names(x))){
      nested_list <- x
      all_r <- x[[1]]$r
      spacing <- diff(sort(all_r))
      threshold <- median(spacing) * 1.5
      
      rngs <- list()
      for (i in seq_along(nested_list)) {
        ge <- nested_list[[i]]
        df <- as.data.frame(ge)
        
        low_r  <- df$r[df$obs < df$lo]
        high_r <- df$r[df$obs > df$hi]
        
        a_l <- get_timestamp_ranges(low_r, threshold)
        a_h <- get_timestamp_ranges(high_r, threshold)
        
        combined <- rbind(
          if(!is.null(a_l) && nrow(a_l) > 0) cbind(a_l, rep("Lower", nrow(a_l)), rep(i, nrow(a_l))) else NULL,
          if(!is.null(a_h) && nrow(a_h) > 0) cbind(a_h, rep("Upper", nrow(a_h)), rep(i, nrow(a_h))) else NULL
        )
        
        if(!is.null(combined) && nrow(combined) > 0) {
          rngs[[i]] <- combined
        }
      }
      
      rngs_df <- do.call(rbind, rngs)
      colnames(rngs_df) <- c("xmin", "xmax", "type", "envelope_id")
      
      data <- data.frame(
        xmin = as.numeric(rngs_df[, "xmin"]),
        xmax = as.numeric(rngs_df[, "xmax"]),
        type = factor(rngs_df[, "type"], levels = c("Lower", "Upper")),
        envelope_id = factor(rngs_df[, "envelope_id"])
      )
      
      data$envelope_id <- factor(data$envelope_id, levels = sort(unique(as.numeric(as.character(data$envelope_id)))))
      
      
      ggplot2::ggplot(data, ggplot2::aes(y = envelope_id, color = type)) +
        ggplot2::geom_errorbarh(ggplot2::aes(xmin = xmin, xmax = xmax), height = 0.4, lwd = 2) +
        ggplot2::scale_color_manual(values = c("Lower" = "black", "Upper" = "grey70")) +
        ggplot2::theme_minimal() +   # <-- unified background
        ggplot2::labs(x = expression(italic(r)), y = "Envelope ID", color = NULL)
      
    }else{
      
      all_data <- lapply(seq_along(x), function(set_idx) {
        nested_list <- x[[set_idx]]
        
        # last_list <- nested_list[[length(nested_list)]]
        # if(inherits(last_list, "global_envelope")) last_list <- list(last_list)
        # 
        all_r <- nested_list[[1]]$r
        spacing <- diff(sort(all_r))
        threshold <- median(spacing) * 1.5
        
        rngs <- list()
        for (i in seq_along(nested_list)) {
          ge <- nested_list[[i]]
          df <- as.data.frame(ge)
          
          low_r  <- df$r[df$obs < df$lo]
          high_r <- df$r[df$obs > df$hi]
          
          a_l <- get_timestamp_ranges(low_r, threshold)
          a_h <- get_timestamp_ranges(high_r, threshold)
          
          combined <- rbind(
            if(!is.null(a_l) && nrow(a_l) > 0) cbind(a_l, rep("Lower", nrow(a_l)), rep(i, nrow(a_l))) else NULL,
            if(!is.null(a_h) && nrow(a_h) > 0) cbind(a_h, rep("Upper", nrow(a_h)), rep(i, nrow(a_h))) else NULL
          )
          
          if(!is.null(combined) && nrow(combined) > 0) {
            rngs[[i]] <- combined
          }
        }
        
        rngs_df <- do.call(rbind, rngs)
        colnames(rngs_df) <- c("xmin", "xmax", "type", "envelope_id")
        
        data <- data.frame(
          xmin = as.numeric(rngs_df[, "xmin"]),
          xmax = as.numeric(rngs_df[, "xmax"]),
          type = factor(rngs_df[, "type"], levels = c("Lower", "Upper")),
          envelope_id = factor(rngs_df[, "envelope_id"]),
          set = factor(set_idx)
        )
        
        data$envelope_id <- factor(data$envelope_id, levels = sort(unique(as.numeric(as.character(data$envelope_id)))))
        
        return(data)
      })
      
      all_data <- do.call(rbind, all_data)
      
      ggplot2::ggplot(all_data, ggplot2::aes(y = envelope_id, color = type)) +
        ggplot2::geom_errorbarh(ggplot2::aes(xmin = xmin, xmax = xmax), height = 0.4, lwd = 2) +
        ggplot2::scale_color_manual(values = c("Lower" = "black", "Upper" = "grey70")) +
        ggplot2::theme_minimal() +   # <-- unified background
        ggplot2::labs(x = expression(italic(r)), y = "Envelope ID", color = NULL) +
        ggplot2::facet_wrap(~set, scales = "free_y")
      
    }
    
    
  }
}

utils::globalVariables(c(
  "r", "lo", "hi", "obs", "central",
  "xmin", "xmax", "type", "envelope_id"
))

