% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/splineplot.R
\name{splineplot}
\alias{splineplot}
\title{Spline Plot for GAM and GLM Models}
\usage{
splineplot(
  fit,
  data,
  xvar = NULL,
  by_var = NULL,
  refx = NULL,
  term_index = 1,
  bins = 12,
  xlim = NULL,
  ylim = NULL,
  show_hist = NULL,
  log_scale = FALSE,
  ci_level = 0.95,
  show_ref_point = TRUE,
  colors = NULL,
  ribbon_ci = FALSE,
  xlab = NULL,
  ylab = NULL,
  ylab_right = "Percent of Population"
)
}
\arguments{
\item{fit}{A fitted model object (gam, glm, lm, coxph)}

\item{data}{The data frame used to fit the model}

\item{xvar}{Character string specifying the variable name for x-axis (default: first spline term)}

\item{by_var}{Character string specifying the interaction variable (default: auto-detect from model)}

\item{refx}{Reference value for the x variable (default: median)}

\item{term_index}{For GAM with multiple smooth terms, which term to plot (default: 1)}

\item{bins}{Number of bins for histogram (default: 12)}

\item{xlim}{X-axis limits (default: range of x variable)}

\item{ylim}{Y-axis limits (default: auto-determined, e.g., c(0.25, 2.0) for HR/OR/RR)}

\item{show_hist}{Logical, whether to show histogram (default: TRUE)}

\item{log_scale}{Logical, whether to use log scale for OR/RR/HR (default: FALSE)}

\item{ci_level}{Confidence interval level (default: 0.95)}

\item{show_ref_point}{Logical, whether to show reference point marker (default: TRUE)}

\item{colors}{Named vector of colors for by_var levels}

\item{ribbon_ci}{Logical, whether to use ribbon style for CI (default: FALSE, uses dotted lines)}

\item{xlab}{Custom x-axis label (default: xvar name)}

\item{ylab}{Custom y-axis label (default: auto-determined based on model type)}

\item{ylab_right}{Custom right y-axis label for histogram (default: "Percent of Population")}
}
\value{
A ggplot2 object
}
\description{
Create ggplot2 visualizations of smooth or spline effects from GAM and GLM models.
Supports Linear, Logistic, Poisson, and Cox models with interaction terms.
Handles GAM smooth terms (s(), te(), ti()), GLM splines (ns(), bs()), and Cox pspline().
}
\examples{
# Create sample data
set.seed(123)
n <- 200
x <- rnorm(n, mean = 50, sd = 10)
lp <- -0.05*(x - 50) + 0.001*(x - 50)^2
y <- rbinom(n, 1, plogis(lp))
dat <- data.frame(x = x, y = y)

# GLM with natural splines
library(splines)
fit_glm <- glm(y ~ ns(x, df = 4), family = binomial(), data = dat)
p <- splineplot(fit_glm, dat)

\donttest{
# GAM example (requires mgcv)
if (requireNamespace("mgcv", quietly = TRUE)) {
  fit_gam <- mgcv::gam(y ~ s(x), family = binomial(), data = dat)
  p2 <- splineplot(fit_gam, dat)
}

# Cox model example (requires survival)
if (requireNamespace("survival", quietly = TRUE)) {
  time <- rexp(n, rate = exp(lp/2))
  status <- rbinom(n, 1, 0.8)
  dat$time <- time
  dat$status <- status
  fit_cox <- survival::coxph(survival::Surv(time, status) ~ ns(x, df = 4),
                              data = dat)
  p3 <- splineplot(fit_cox, dat)
}
}
}
